/** @format */
/** @format */
const { ObjectId } = require('mongoose').Types;

const {
  createDoc,
  getDocs,
  updateDoc,
  deleteDoc
} = require('@helpers/factoryFN');

const _ = require('lodash');
const { User } = require('../../user/model/user');
const { _responseWrapper } = require('../../../helpers/util-response');



exports.getMyProfile = async (req) => {
  let agg = [
    {
      '$match': {
        '_id': new ObjectId(req.userId)
      }
    }, {
      '$lookup': {
        'from': 'posts',
        'localField': '_id',
        'foreignField': 'user',
        'as': 'posts'
      }
    }, {
      '$project': {
        'name': 1,
        'avatar': '$avatar.url',
        'role': 1,
        'posts': {
          '$map': {
            'input': '$posts',
            'as': 'post',
            'in': {
              '_id': '$$post._id',
              'url': {
                '$arrayElemAt': [
                  '$$post.media.url', 0
                ]
              }
            }
          }
        }
      }
    }
  ]

  let data = await getDocs(User, { agg })(req)
  data.response.data = data.response.data[0]
  return data
}

exports.getProfile = async (req) => {
  try {
    let agg = [
      {
        '$match': {
          '_id': new ObjectId(req.params.id)
        }
      }, {
        '$lookup': {
          'from': 'posts',
          'localField': '_id',
          'foreignField': 'user',
          'as': 'posts'
        }
      }, {
        '$project': {
          'name': 1,
          'avatar': '$avatar.url',
          'role': 1,
          'posts': {
            '$map': {
              'input': '$posts',
              'as': 'post',
              'in': {
                '_id': '$$post._id',
                'url': {
                  '$arrayElemAt': [
                    '$$post.media.url', 0
                  ]
                }
              }
            }
          }
        }
      }
    ]
    let data = await getDocs(User, { agg })({ ...req, params: {} })
    data.response.data = data.response.data[0]
    return data
  } catch (error) {
    console.log(error)
  }
}

exports.followUser = async (req) => {
  try {
    const { userId } = req; // Requesting user's ID
    const { userId: targetUserId } = req.body; // ID of the user to follow

    // Check if user is trying to follow themselves
    if (userId === targetUserId) {
      return _responseWrapper(true, 'Cannot follow yourself', 400);
    }

    // Check if the user is already following the target user
    const existingFollowing = await getDocs(User, {
      query: { followings: { $in: [targetUserId] } }
    })({ ...req, params: { id: userId } });

    if (existingFollowing.response.data) {
      return _responseWrapper(true, 'Already Following', 200);
    }

    // Add userId to the followers of the target user
    const updateTargetUser = await updateDoc(User, {
      updateObject: { update: { $push: { followers: userId } } }
    })({ ...req, body: {}, params: { id: targetUserId } });

    // Add targetUserId to the followings of the current user
    const updateCurrentUser = await updateDoc(User, {
      updateObject: { update: { $push: { followings: targetUserId } } }
    })({ ...req, body: {}, params: { id: userId } });

    // If both updates are successful
    if (updateTargetUser.statusCode === 200 && updateCurrentUser.statusCode === 200) {
      return _responseWrapper(true, 'Successfully Followed', 200);
    }

    // Handle failure cases (fallback if any operation fails)
    return _responseWrapper(false, 'Failed to follow user', 500);
  } catch (error) {
    console.error('Error in followUser:', error);
    return _responseWrapper(false, 'An error occurred while following user', 500);
  }
};


exports.unfollowUser = async (req) => {
  try {
    const { userId } = req; // Requesting user's ID
    const { userId: targetUserId } = req.body; // ID of the user to unfollow

    // Check if user is trying to unfollow themselves
    if (userId === targetUserId) {
      return _responseWrapper(true, 'Cannot unfollow yourself', 400);
    }

    // Check if the user is not following the target user
    const existingFollowing = await getDocs(User, {
      query: { followings: { $in: [targetUserId] } }
    })({ ...req, params: { id: userId } });

    if (!existingFollowing.response.data) {
      return _responseWrapper(true, 'Not following the user', 200);
    }

    // Remove userId from the followers of the target user
    const updateTargetUser = await updateDoc(User, {
      updateObject: { update: { $pull: { followers: userId } } }
    })({ ...req, body: {}, params: { id: targetUserId } });

    // Remove targetUserId from the followings of the current user
    const updateCurrentUser = await updateDoc(User, {
      updateObject: { update: { $pull: { followings: targetUserId } } }
    })({ ...req, body: {}, params: { id: userId } });

    // If both updates are successful
    if (updateTargetUser.statusCode === 200 && updateCurrentUser.statusCode === 200) {
      return _responseWrapper(true, 'Successfully Unfollowed', 200);
    }

    // Handle failure cases (fallback if any operation fails)
    return _responseWrapper(false, 'Failed to unfollow user', 500);
  } catch (error) {
    console.error('Error in unfollowUser:', error);
    return _responseWrapper(false, 'An error occurred while unfollowing user', 500);
  }
};

exports.getFollowers = async (req) => {
  try {
    const { userId } = req; // Requesting user's ID

    const followers = await getDocs(User, {
      query: { followings: { $in: [userId] } },
      select: 'name avatar'
    })({ ...req, params: {} });

    return followers;
  } catch (error) {
    console.error('Error in getFollowers:', error);
    return _responseWrapper(false, 'An error occurred while fetching followers', 500);
  }
}

exports.getFollowings = async (req) => {
  try {
    const { userId } = req; // Requesting user's ID

    const following = await getDocs(User, {
      query: { followers: { $in: [userId] } },
      select: 'name avatar'
    })({ ...req, params: {} });

    return following;
  } catch (error) {
    console.error('Error in getFollowing:', error);
    return _responseWrapper(false, 'An error occurred while fetching following', 500);
  }
}

